/* 
  Current Transformer Application Note
  Name: current_cloud.ino
  Purpose: Interfaces with SCT013-000, which is non-invasive current transfomer to measure Current, to Portenta C33 and send RMS Current informatio to Arduino Cloud.
*/

#include "thingProperties.h"

#define Sensor_Factor  51.8
#define Region_Voltage 110
#define Read_APin      A0

void setup() {
  // Initialize serial and wait for port to open:
  Serial.begin(115200);
  pinMode(Read_APin, INPUT);
  analogReadResolution(12);

  delay(1000); 

  // Defined in thingProperties.h
  initProperties();

  // Connect to Arduino IoT Cloud
  ArduinoCloud.begin(ArduinoIoTPreferredConnection);
  
  /*
     The following function allows you to obtain more information
     related to the state of network and IoT Cloud connection and errors
     the higher number the more granular information you’ll get.
     The default is 0 (only errors).
     Maximum is 4
 */
  setDebugMessageLevel(2);
  ArduinoCloud.printDebugInfo();
}

void loop() {
  // Cloud update
  ArduinoCloud.update();
  
  float Irms = getCurrent();                                         // Function to obtain RMS Current
  float P = Irms*Region_Voltage;                                     // P = IV (Watts) | Apparent Power [kVA]
  
  // Update Current value
  cloud_Current = Irms;

  Serial.print(F("I_rms [A]: "));
  Serial.print(Irms);
  Serial.print(F("Power [W]: "));
  Serial.print(P);
  delay(100); 
}

/**
  Reads the designated analog pin, interfaced to a SCT013-000 sensor via interpreter circuit, and obtains RMS Current.

  @param Current represents the RMS Current
  @param sensorValue value read from analog pin with SCT013-000 sensor
  @param I_Sum sum of RMS Current value and is compensated afterwards for negative semi-cycle cuadratics
  @param N counter that follows the amount of retrieved current data
  @return the RMS Current
*/
float getCurrent(){
  float Current = 0;
  float I_Sum = 0;
  int N=0;
  long time = millis();

  while(millis() - time<500){                                       // 30 cycles of 60 Hz approx. for samples collected in 0.5 seconds of runtime
    
    int sensorValue = analogRead(Read_APin);                        // Input pin read
    float sensorVoltage = sensorValue * (3.1 / 4096.0);             // ADC with the full 0 – 3.1 V range

    /** 
      Current = V_sense*(100A/1.65V), using 33 Ohms of external load resistance
      factor of 66.67 for 30 Ohm configuration; ~51.8 for 39 Ohm configuration
    */
    Current = sensorVoltage * Sensor_Factor;
    I_Sum += sq(Current);                                           
    N++;
    delay(1);
  }

  I_Sum = I_Sum * 2;                                                // Negative semi-cycle cuadratics compensation
  Current = sqrt((I_Sum)/N);                                        // Average RMS current
  return(Current);
}

/*
  Since CloudCurrent is READ_WRITE variable, onCloudCurrentChange() is
  executed every time a new value is received from IoT Cloud.
*/
void onCloudCurrentChange()  {
  // Add your code here to act upon CloudCurrent change
  // Simple debugging message. Can be disabled
  Serial.print(F("Updated I_rms value for Cloud"));
}