/**
  Smart Irrigation System using the Arduino Edge Control - Application Note
  Name: Edge_Control_Code.ino
  Purpose: 4 Zones Smart Irrigation System using the Arduino Edge Control with Cloud connectivity using a MKR WiFi 1010.

  @author Christopher Mendez
*/

#include "Helpers.h"
#include <Arduino_EdgeControl.h>
#include <Wire.h>
#include "SensorValues.hpp"

// The MKR1 board I2C address
#define EDGE_I2C_ADDR 0x05

/** UI Management **/
// Button statuses
enum ButtonStatus : byte {
  ZERO_TAP,
  SINGLE_TAP,
  DOUBLE_TAP,
  TRIPLE_TAP,
  QUAD_TAP,
  FIVE_TAP,
  LOT_OF_TAPS
};

// ISR: count the button taps
volatile byte taps{ 0 };
// ISR: keep elapsed timings
volatile unsigned long previousPress{ 0 };
// ISR: Final button status
volatile ButtonStatus buttonStatus{ ZERO_TAP };


constexpr unsigned int adcResolution{ 12 };  // Analog Digital Converter resolution for the 4-20mA sensor.

//4-20mA input channel definition
constexpr pin_size_t inputChannels[]{
  INPUT_420mA_CH01,
};

constexpr size_t inputChannelsLen{ sizeof(inputChannels) / sizeof(inputChannels[0]) };
int inputChannelIndex{ 0 };

//4-20mA average function structure
struct Voltages {
  float volt3V3;
  float voltRef;
};

unsigned long previousMillis = 0;  // will store last time the sensors were updated

const long interval = 1000;  // interval at which to update sensores (milliseconds)

// Valves flow control variables
bool controlV1 = 1;
bool controlV2 = 1;
bool controlV3 = 1;
bool controlV4 = 1;

// LCD flow control variables
bool controlLCD = 1;
int showTimeLCD = 0;

// Valves On time kepping variables
int StartTime1, CurrentTime1;
int StartTime2, CurrentTime2;
int StartTime3, CurrentTime3;
int StartTime4, CurrentTime4;

SensorValues_t vals;

/**
  Main section setup
*/
void setup() {
  EdgeControl.begin();
  Wire.begin();
  delay(500);
  Serial.begin(115200);

  Serial.println("Init begin");

  // Enable power lines
  Power.enable3V3();
  Power.enable5V();
  Power.on(PWR_3V3);
  Power.on(PWR_VBAT);
  Power.on(PWR_MKR1);
  delay(5000);
  Power.on(PWR_19V);

  // Init Edge Control IO Expander
  Serial.print("IO Expander initializazion ");
  if (!Expander.begin()) {
    Serial.println("failed.");
    Serial.println("Please, be sure to enable gated 3V3 and 5V power rails");
    Serial.println("via Power.enable3V3() and Power.enable5V().");
  } else Serial.println("succeeded.");

  // LCD button definition
  pinMode(POWER_ON, INPUT);
  attachInterrupt(POWER_ON, buttonPress, RISING);

  // Arduino Edge Control ports init
  Input.begin();
  Input.enable();
  Latching.begin();

  analogReadResolution(adcResolution);

  setSystemClock(__DATE__, __TIME__);  // define system time as a reference for the RTC

  //RealTimeClock.setEpoch(1684803771-(3600*4));  // uncomment this to set the RTC time once.

  // Init the LCD display
  LCD.begin(16, 2);
  LCD.backlight();

  LCD.home();
  LCD.print("Smart Irrigation");
  LCD.setCursor(5, 1);
  LCD.print("System");
  delay(2000);

  LCD.clear();
}

void loop() {
  // LCD button taps detector function
  detectTaps();

  // Different button taps handler
  switch (buttonStatus) {
    case ZERO_TAP:  // will execute always the button is not being pressed.
      if (controlLCD == 1 && showTimeLCD == 0) {
        ValvesStatusLCD();  // when there is not an active timer
        controlLCD = 0;
      }

      if (showTimeLCD == 1) {
        ValvesTimersLCD();  // when there is an active timer
      }

      break;

    case SINGLE_TAP:  // will execute when the button is pressed once.
      Serial.println("Single Tap");
      vals.z1_local = !vals.z1_local;
      sendValues(&vals);
      buttonStatus = ZERO_TAP;
      break;

    case DOUBLE_TAP:  // will execute when the button is pressed twice.
      Serial.println("Double Tap");
      vals.z2_local = !vals.z2_local;
      sendValues(&vals);
      buttonStatus = ZERO_TAP;
      break;

    case TRIPLE_TAP:  // will execute when the button is pressed three times.
      Serial.println("Triple Tap");
      vals.z3_local = !vals.z3_local;
      sendValues(&vals);
      buttonStatus = ZERO_TAP;
      break;

    case QUAD_TAP:  // will execute when the button is pressed four times.
      Serial.println("Quad Tap");
      vals.z4_local = !vals.z4_local;
      sendValues(&vals);
      buttonStatus = ZERO_TAP;
      break;

    case FIVE_TAP:  // will execute when the button is pressed five times.
      Serial.println("Five Tap");
      LCD.backlight();
      LCD.home();
      
      break;

    default:
      Serial.println("Too Many Taps");
      buttonStatus = ZERO_TAP;
      break;
  }

  // reset the valves accumuldated on time every day at midnight
  if (getLocalhour() == " 00:00:00") {
    Serial.println("Resetting accumulators every day");
    vals.z1_on_time = 0;
    vals.z2_on_time = 0;
    vals.z3_on_time = 0;
    vals.z4_on_time = 0;
    delay(1000);
  }

  unsigned long currentMillis = millis();

  if (currentMillis - previousMillis >= interval) {

    previousMillis = currentMillis;

    // send local sensors values and retrieve cloud variables status back and forth
    updateSensors();
  }

  // activate, deactive and keep time of valves function
  valvesHandler();
}

/**
  Poor-man LCD button debouncing function. 
*/
void buttonPress() {
  const auto now = millis();

  if (now - previousPress > 100)
    taps++;

  previousPress = now;
}

/**
  Detect and count button taps
*/
void detectTaps() {
  // Timeout to validate the button taps counter
  constexpr unsigned int buttonTapsTimeout{ 300 };

  // Set the button status and reset the taps counter when button has been
  // pressed at least once and button taps validation timeout has been reached.
  if (taps > 0 && millis() - previousPress >= buttonTapsTimeout) {
    buttonStatus = static_cast<ButtonStatus>(taps);
    taps = 0;
  }
}

/**
  This function retrieve data from the MKR to update the local valves status, also reads the water level sensor
  and send back the valves status if there was any manual change to the MKR to be updated in the cloud.
*/
void updateSensors() {

  // function that request valves status from the MKR through I2C
  getCloudValues();

  auto [voltsMuxer, voltsReference] = getAverageAnalogRead(inputChannels[inputChannelIndex]);

  inputChannelIndex = ++inputChannelIndex % inputChannelsLen;
  float w_level = ((voltsReference / 220.0 * 1000.0) - 4.0) * 6.25;
  
  if (w_level > 0) {
    vals.water_level_local = w_level;
  } else {
    vals.water_level_local = 0;
  }

  vals.z1_on_time;
  vals.z2_on_time;
  vals.z3_on_time;
  vals.z4_on_time;

  sendValues(&vals);

  // Uncomment this function to read in the serial monitor the values received by the I2C communication
  //logValues(&vals);
}

/**
  Debugging function to print the variables status being sent and read from the MKR

  @param values The I2C communicated sensors values
*/
void logValues(SensorValues_t *values) {
  char telem_buf[200];
  sprintf(telem_buf, "Zone 1: %d, Zone 2: %d, Zone 3: %d, Zone 4: %d, Timer 1: %d,Timer 2: %d,Timer 3: %d,Timer 4: %d, Water Level: %.2f",
          values->z1_local, values->z2_local, values->z3_local, values->z4_local, values->z1_count, values->z2_count, values->z3_count, values->z4_count, values->water_level_local);

  Serial.println(telem_buf);
}

/**
  Function to request a valves and variables update from the MKR.
*/

void getCloudValues() {

  Wire.beginTransmission(EDGE_I2C_ADDR);  //slave's 7-bit I2C address

  // asking if the MKR board is connected to be able to communicate
  byte busStatus = Wire.endTransmission();

  if (busStatus != 0) {  // if the slave is not connected this happens

  } else {  // if the slave is connected this happens

    Wire.requestFrom(EDGE_I2C_ADDR, sizeof(SensorValues_t));

    uint8_t buf[200];
    for (uint8_t i = 0; i < sizeof(SensorValues_t); i++)
      if (Wire.available())
        buf[i] = Wire.read();

    SensorValues_t *values = (SensorValues_t *)buf;

    vals.z1_local = values->z1_local;
    vals.z2_local = values->z2_local;
    vals.z3_local = values->z3_local;
    vals.z4_local = values->z4_local;

    if (values->z1_count > 0 || values->z2_count > 0 || values->z3_count > 0 || values->z4_count > 0) {
      Serial.println("There's an active timer");
      showTimeLCD = 1;  // instead of showing valves status in the LCD, shows the countdown timers.
    } else {
      Serial.println("There's not an active timer");
      showTimeLCD = 0;  // show the valves satus in the LCD.
    }

    // convert seconds to minutes (these variables store the countdown timers)
    vals.z1_count = values->z1_count / 60.0;
    vals.z2_count = values->z2_count / 60.0;
    vals.z3_count = values->z3_count / 60.0;
    vals.z4_count = values->z4_count / 60.0;
  }
}

/**
  Function that sends the local sensors values through I2C to the MKR
  @param values The I2C communicated sensors values
*/
void sendValues(SensorValues_t *values) {
  writeBytes((uint8_t *)values, sizeof(SensorValues_t));
}

/**
  Function that transport the sensors data through I2C to the MKR
  @param buf store the structured sensors values
  @param len store the buffer lenght
*/
void writeBytes(uint8_t *buf, uint8_t len) {

  Wire.beginTransmission(EDGE_I2C_ADDR);

  for (uint8_t i = 0; i < len; i++) {
    Wire.write(buf[i]);
  }

  Wire.endTransmission();
}

/**
  Function that reads the valves values received from the MKR and controls them,
  measures the ON time of each one and update their status on the LCD screen.
*/
void valvesHandler() {

  if (vals.z1_local == 1 && controlV1 == 1) {
    Serial.println("Opening Valve 1");
    LCD.clear();
    LCD.setCursor(1, 0);
    LCD.print("Opening Valve");
    LCD.setCursor(7, 1);
    LCD.print("#1");
    controlLCD = 1;
    Latching.channelDirection(LATCHING_OUT_1, NEGATIVE);
    Latching.strobe(4500);
    StartTime1 = time(NULL);
    controlV1 = 0;
    LCD.clear();
  } else if (vals.z1_local == 0 && controlV1 == 0) {
    Serial.println("Closing Valve 1");
    LCD.clear();
    LCD.setCursor(1, 0);
    LCD.print("Closing Valve");
    LCD.setCursor(7, 1);
    LCD.print("#1");
    controlLCD = 1;
    Latching.channelDirection(LATCHING_OUT_1, POSITIVE);
    Latching.strobe(4500);
    CurrentTime1 = time(NULL);
    Serial.print("V1 On Time: ");
    vals.z1_on_time += (CurrentTime1 - StartTime1) / 60.0;
    Serial.println(vals.z1_on_time);
    controlV1 = 1;
    LCD.clear();
  }

  if (vals.z2_local == 1 && controlV2 == 1) {
    Serial.println("Opening Valve 2");
    LCD.clear();
    LCD.setCursor(1, 0);
    LCD.print("Opening Valve");
    LCD.setCursor(7, 1);
    LCD.print("#2");
    controlLCD = 1;
    Latching.channelDirection(LATCHING_OUT_3, NEGATIVE);
    Latching.strobe(4500);
    StartTime2 = time(NULL);
    controlV2 = 0;
    LCD.clear();
  } else if (vals.z2_local == 0 && controlV2 == 0) {
    Serial.println("Closing Valve 2");
    LCD.clear();
    LCD.setCursor(1, 0);
    LCD.print("Closing Valve");
    LCD.setCursor(7, 1);
    LCD.print("#2");
    controlLCD = 1;
    Latching.channelDirection(LATCHING_OUT_3, POSITIVE);
    Latching.strobe(4500);
    CurrentTime2 = time(NULL);
    Serial.print("V2 On Time: ");
    vals.z2_on_time += (CurrentTime2 - StartTime2) / 60.0;
    Serial.println(vals.z2_on_time);
    controlV2 = 1;
    LCD.clear();
  }

  if (vals.z3_local == 1 && controlV3 == 1) {
    Serial.println("Opening Valve 3");
    LCD.clear();
    LCD.setCursor(1, 0);
    LCD.print("Opening Valve");
    LCD.setCursor(7, 1);
    LCD.print("#3");
    controlLCD = 1;
    Latching.channelDirection(LATCHING_OUT_5, NEGATIVE);
    Latching.strobe(4500);
    StartTime3 = time(NULL);
    controlV3 = 0;
    LCD.clear();
  } else if (vals.z3_local == 0 && controlV3 == 0) {
    Serial.println("Closing Valve 3");
    LCD.clear();
    LCD.setCursor(1, 0);
    LCD.print("Closing Valve");
    LCD.setCursor(7, 1);
    LCD.print("#3");
    controlLCD = 1;
    Latching.channelDirection(LATCHING_OUT_5, POSITIVE);
    Latching.strobe(4500);
    CurrentTime3 = time(NULL);
    Serial.print("V3 On Time: ");
    vals.z3_on_time += (CurrentTime3 - StartTime3) / 60.0;
    Serial.println(vals.z3_on_time);
    controlV3 = 1;
    LCD.clear();
  }

  if (vals.z4_local == 1 && controlV4 == 1) {
    Serial.println("Opening Valve 4");
    LCD.clear();
    LCD.setCursor(1, 0);
    LCD.print("Opening Valve");
    LCD.setCursor(7, 1);
    LCD.print("#4");
    controlLCD = 1;
    Latching.channelDirection(LATCHING_OUT_7, NEGATIVE);
    Latching.strobe(4500);
    StartTime4 = time(NULL);
    controlV4 = 0;
    LCD.clear();
  } else if (vals.z4_local == 0 && controlV4 == 0) {
    Serial.println("Closing Valve 4");
    LCD.clear();
    LCD.setCursor(1, 0);
    LCD.print("Closing Valve");
    LCD.setCursor(7, 1);
    LCD.print("#4");
    controlLCD = 1;
    Latching.channelDirection(LATCHING_OUT_7, POSITIVE);
    Latching.strobe(4500);
    CurrentTime4 = time(NULL);
    Serial.print("V4 On Time: ");
    vals.z4_on_time += (CurrentTime4 - StartTime4) / 60.0;
    Serial.println(vals.z4_on_time);
    controlV4 = 1;
    LCD.clear();
  }
}

/**
  Average the water level sensor readings to improve stability
  @param pin The input pin where the 4-20mA sensor is connected
*/
Voltages getAverageAnalogRead(int pin) {
  constexpr size_t loops{ 100 };
  constexpr float toV{ 3.3f / float{ (1 << adcResolution) - 1 } };
  constexpr float rDiv{ 17.4f / (10.0f + 17.4f) };

  int tot{ 0 };

  for (auto i = 0u; i < loops; i++)
    tot += Input.analogRead(pin);
  const auto avg = static_cast<float>(tot) * toV / static_cast<float>(loops);

  return { avg, avg / rDiv };
}

/**
  Function that shows each valve current status on the LCD screen.
*/
void ValvesStatusLCD() {
  String msg, msg2;

  msg = "Z1:";
  msg += vals.z1_local;
  msg2 = " Z2:";
  msg2 += vals.z2_local;
  msg2 += " Z3:";
  msg2 += vals.z3_local;
  msg2 += " Z4:";
  msg2 += vals.z4_local;

  //LCD.clear();
  LCD.setCursor(1, 0);
  LCD.print("Valves: ");
  LCD.print(msg);
  LCD.setCursor(0, 1);
  LCD.print(msg2);
}

/**
  Function that shows each valve current countdown timer on the LCD screen.
*/
void ValvesTimersLCD() {

  char line1[16];
  char line2[16];

  //LCD.clear();

  float z1Time = 0;
  float z2Time = 0;
  float z3Time = 0;
  float z4Time = 0;

  if (vals.z1_count > 0) {
    z1Time = vals.z1_count - ((time(NULL) - StartTime1) / 60.0);
  } else {
    z1Time = 0;
  }

  if (vals.z2_count > 0) {
    z2Time = vals.z2_count - ((time(NULL) - StartTime2) / 60.0);
  } else {
    z2Time = 0;
  }

  if (vals.z3_count > 0) {
    z3Time = vals.z3_count - ((time(NULL) - StartTime3) / 60.0);
  } else {
    z3Time = 0;
  }

  if (vals.z4_count > 0) {
    z4Time = vals.z4_count - ((time(NULL) - StartTime4) / 60.0);
  } else {
    z4Time = 0;
  }

  sprintf(line1, "Z1:%.1f'  Z2:%.1f'    ", z1Time, z2Time);  // use sprintf() to compose the string line1
  sprintf(line2, "Z3:%.1f'  Z4:%.1f'    ", z3Time, z4Time);  // use sprintf() to compose the string line2

  if ((time(NULL) - StartTime1) / 60.0 <= vals.z1_count || (time(NULL) - StartTime2) / 60.0 <= vals.z2_count || (time(NULL) - StartTime3) / 60.0 <= vals.z3_count || (time(NULL) - StartTime4) / 60.0 <= vals.z4_count) {

    LCD.setCursor(0, 0);
    LCD.print(line1);
    LCD.setCursor(0, 1);
    LCD.print(line2);
  }
}
