/**
  Door Intruder Detector using ML with the Nicla Voice - Application Note
  Name: Portenta_H7_Code.ino
  Purpose: Portenta H7 as host receiving BLE notifications from a Nicla Voice and sending them to the Arduino IoT Cloud for visualization.

  @author Christopher Mendez
*/

#include <ArduinoIoTCloud.h>
#include <Arduino_ConnectionHandler.h>
#include "thingProperties.h"
#include <ArduinoBLE.h>

#define DEBUG false

// Global Parameters
byte BatteryValue = 0;  // last battery level received.
byte AlertValue = 0;    // last alert value received.
bool control = false;   // flow control variable for repetitive events

/**
  Main section setup
*/
void setup() {

  // Initialize serial and wait for port to open:
  Serial.begin(115200);

  // Arduino Cloud connection status LED
  pinMode(LEDG, OUTPUT);

  // Initialize BLE
  if (!BLE.begin()) {
    Serial.println("Starting BLE failed!");
    while (1) {
    }
  }

#ifndef TARGET_PORTENTA_H7
  Serial.println("Unsupported board!");
  while (1)
    ;
#endif

  // Defined in thingProperties.h
  initProperties();

  // Connect to Arduino IoT Cloud
  if (!ArduinoCloud.begin(ArduinoIoTPreferredConnection)) {
    Serial.println("ArduinoCloud.begin FAILED!");
  }

  setDebugMessageLevel(2);
  ArduinoCloud.printDebugInfo();

  // start scanning for peripheral
  BLE.scan();
  Serial.println("Scanning for peripherals.");
}

void loop() {

  // check if a peripheral has been discovered
  BLEDevice peripheral = BLE.available();

  // Turn on the Green LED if the board is successfully connected to the Cloud. On with Low.
  if (ArduinoCloud.connected()) {
    digitalWrite(LEDG, LOW);
  } else {
    digitalWrite(LEDG, HIGH);
  }

  if (peripheral) {
    // peripheral discovered, print out address, local name, and advertised service
    Serial.print("Found ");
    Serial.print(peripheral.address());
    Serial.print(" '");
    Serial.print(peripheral.localName());
    Serial.print("' ");
    Serial.print(peripheral.advertisedServiceUuid());
    Serial.println();

    // Check if the peripheral is a Nicla Lock:
    if (peripheral.localName() == "Nicla Lock") {
      // stop scanning
      BLE.stopScan();

      // Nicla Voice node connection handler
      NiclaLockHandler(peripheral);

      // peripheral disconnected, start scanning again
      BLE.scan();
    }
  }
  ArduinoCloud.update();
}

/*************************************
* Portenta H7 host Routines
*************************************/


/**
  Read the battery level from the battery charger IC 

  @param peripheral Nicla Lock peripheral and all it's characteristics.
  @param batteryLevelChar standard characteristic for battery level.
  @param alertLevel standard characteristic for inmediate alerts.
*/
void NiclaLockHandler(BLEDevice peripheral) {
  // connect to the peripheral
  Serial.println("Connecting ...");
  if (peripheral.connect()) {
    Serial.println("Connected");
  } else {
    Serial.println("Failed to connect!");
    return;
  }

  // discover peripheral attributes
  Serial.println("Searching for service 1802 ...");

  if (peripheral.discoverService("1802")) {
    Serial.println("Service discovered");
  } else {
    Serial.println("Attribute discovery failed.");
    peripheral.disconnect();

    while (1)
      ;
    return;
  }

  // retrieve the simple key characteristics
  BLECharacteristic batteryLevelChar = peripheral.characteristic("2A19");
  BLECharacteristic alertLevel = peripheral.characteristic("2A06");

  // subscribe to the simple key characteristics process
  Serial.println("Subscribing to simple key characteristic ...");
  if (!batteryLevelChar || !alertLevel) {
    Serial.println("no simple key characteristic found!");
    peripheral.disconnect();
    return;
  } else if (!batteryLevelChar.canSubscribe() || !alertLevel.canSubscribe()) {
    Serial.println("simple key characteristic is not subscribable!");
    peripheral.disconnect();
    return;
  } else if (!batteryLevelChar.subscribe() || !alertLevel.subscribe()) {
    Serial.println("subscription failed!");
    peripheral.disconnect();
    return;
  } else {
    Serial.println("Subscribed to Battery Level and Alert Characteristic");
  }

  // Gathering Characteristics initial values and printing them
  batteryLevelChar.readValue(BatteryValue);
  alertLevel.readValue(AlertValue);

  Serial.print("Battery Level: ");
  Serial.print(BatteryValue);
  Serial.println(" %");
  battery = BatteryValue;

  Serial.print("Alert: ");
  Serial.println(AlertValue);
  alertStatus = AlertValue;
  

  // while the peripheral is connected
  while (peripheral.connected()) {
    BLEstatus = true;
    // check if the value of the characteristic has been updated
    if (batteryLevelChar.valueUpdated()) {

      batteryLevelChar.readValue(BatteryValue);

      Serial.print("Battery Level: ");
      Serial.print(BatteryValue);
      Serial.println(" %");

      battery = BatteryValue;
    }

    // check if the value of the characteristic has been updated
    if (alertLevel.valueUpdated()) {

      alertLevel.readValue(AlertValue);

      Serial.print("Alert: ");
      Serial.println(AlertValue);

      
      if (AlertValue == 1) {    // if the Alert = 1 means the door was opened
        control = !control;     // as variables are updated if they change, add a simple dot "." to be able to update it if happens twice consecutive.
        if (control) {
          DoorEvent = "Door opened";
        } else {
          DoorEvent = "Door opened.";
        }

        alertStatus = false;

      } else if (AlertValue == 2) {   // if the Alert = 2 means the door was forced
        control = !control;           // as variables are updated if they change, add a simple dot "." to be able to update it if happens twice consecutive.
        if (control) {
          DoorEvent = "Intruder detected!";
        } else {
          DoorEvent = "Intruder detected!.";
        }

        alertStatus = true;
      }
    }

    // turn off green LED if disconnected from Cloud.
    if (ArduinoCloud.connected()) {
      digitalWrite(LEDG, LOW);
    } else {
      digitalWrite(LEDG, HIGH);
    }

    ArduinoCloud.update();
  }
  
  Serial.println("Nicla Lock disconnected!");
  BLEstatus = false;
  ArduinoCloud.update();
}
